# ============================================================
# Home Data for ML Course (Kaggle Learn Users)
# CatBoost（RMSE学習 + log1p）最終調整：depth=6 を狙う版
#
# 目的：
#   - Public LB の RMSE（小さいほど良い）を、いまのベスト(13135)から
#     さらに少しでも下げる可能性を狙う
#
# 背景：
#   - あなたの結果では depth=7 が強かった
#   - ここからの改善は「過学習をほんの少し抑える」方向が刺さることがある
#   - その代表が depth=6
#
# このコードの設計（重要ポイント）：
#   1) 目的変数は log1p(SalePrice) で学習し、予測は expm1 でドルに戻す
#      → 高額物件の影響を和らげ、RMSEが下がることがある
#   2) CatBoost の強み（カテゴリを直接扱える）を活かす：One-Hot不要
#   3) 欠損処理は「最低限の堅牢化」
#      - object欠損 → "Missing"
#      - 数値欠損 → train中央値で埋める
#   4) 外れ値除去（超広いのに安い）を入れる：RMSE改善の定番
#   5) KFold(5)で学習し、test予測を平均：提出のブレを減らす
#
# 使い方：
#   - Kaggle Notebook の1セルにコピペして実行
#   - 実行後に submission.csv ができるので、そのまま提出
#
# 注意：
#   - 実行時間が長い場合は、iterations/od_wait を下げると短縮できます（下に記載）
# ============================================================

import numpy as np
import pandas as pd

from sklearn.model_selection import KFold
from sklearn.metrics import mean_squared_error

from catboost import CatBoostRegressor, Pool


# ============================================================
# 1) データ読み込み
# ============================================================
train = pd.read_csv("/kaggle/input/home-data-for-ml-course/train.csv")
test  = pd.read_csv("/kaggle/input/home-data-for-ml-course/test.csv")

# ============================================================
# 2) 外れ値除去（定番）
# ------------------------------------------------------------
# GrLivArea が極端に大きいのに SalePrice が低い点が混ざると
# RMSEが悪化しやすいので、経験則で除去することが多い
# ============================================================
train = train.drop(train[(train["GrLivArea"] > 4000) & (train["SalePrice"] < 300000)].index)

# 目的変数
y = train["SalePrice"].copy()

# 特徴量
X = train.drop(columns=["SalePrice"]).copy()

# テスト（予測対象）
X_test = test.copy()


# ============================================================
# 3) 派生特徴量（軽量で効きやすいものだけ）
# ------------------------------------------------------------
# このレベル帯（13000前後）では、派生特徴が「少し効く」ことが多い
# やりすぎると逆にブレることもあるので、定番だけ入れる
# ============================================================
def add_features(df: pd.DataFrame) -> pd.DataFrame:
    df = df.copy()

    # 面積系：欠損は「ない=0」が自然な列が多い（地下室なし等）
    for c in ["TotalBsmtSF", "1stFlrSF", "2ndFlrSF", "GarageArea"]:
        if c in df.columns:
            df[c] = df[c].fillna(0)

    # 合計面積（強い）
    df["TotalSF"] = df.get("TotalBsmtSF", 0) + df.get("1stFlrSF", 0) + df.get("2ndFlrSF", 0)

    # 築年数・改築後年数（強い）
    if "YrSold" in df.columns and "YearBuilt" in df.columns:
        df["HouseAge"] = df["YrSold"] - df["YearBuilt"]
    if "YrSold" in df.columns and "YearRemodAdd" in df.columns:
        df["RemodAge"] = df["YrSold"] - df["YearRemodAdd"]

    # 風呂合計（地味に効く）
    for c in ["FullBath", "HalfBath", "BsmtFullBath", "BsmtHalfBath"]:
        if c in df.columns:
            df[c] = df[c].fillna(0)
    if all(c in df.columns for c in ["FullBath","HalfBath","BsmtFullBath","BsmtHalfBath"]):
        df["TotalBath"] = df["FullBath"] + 0.5 * df["HalfBath"] + df["BsmtFullBath"] + 0.5 * df["BsmtHalfBath"]

    # 品質×面積（強い）
    if "OverallQual" in df.columns and "GrLivArea" in df.columns:
        df["Qual_x_GrLivArea"] = df["OverallQual"] * df["GrLivArea"]
    if "OverallQual" in df.columns:
        df["Qual_x_TotalSF"] = df["OverallQual"] * df["TotalSF"]

    return df

X = add_features(X)
X_test = add_features(X_test)


# ============================================================
# 4) カテゴリ列の指定（CatBoostはこれが重要）
# ------------------------------------------------------------
# CatBoostはカテゴリ列を直接扱えるので、object列をカテゴリとして渡す。
# カテゴリ列は「列番号（index）」で指定できる。
# ============================================================
cat_cols = X.select_dtypes(include=["object"]).columns.tolist()
cat_idx = [X.columns.get_loc(c) for c in cat_cols]
print("Categorical columns:", len(cat_cols))


# ============================================================
# 5) 欠損処理（堅牢化）
# ------------------------------------------------------------
# CatBoostは欠損に強いが、objectのNaNは明示的に文字へ変換した方が安全
# 数値のNaNも、推論時の安定のために中央値で埋める
# ============================================================
# カテゴリ欠損→"Missing"
X[cat_cols] = X[cat_cols].fillna("Missing")
X_test[cat_cols] = X_test[cat_cols].fillna("Missing")

# 数値欠損→中央値（trainで決めた値をtestにも使う）
num_cols = X.columns.difference(cat_cols).tolist()
med = X[num_cols].median()
X[num_cols] = X[num_cols].fillna(med)
X_test[num_cols] = X_test[num_cols].fillna(med)


# ============================================================
# 6) 目的変数の変換：log1p
# ------------------------------------------------------------
# 学習は log1p（価格分布の歪みを弱める）
# 予測は expm1 で元のドルに戻す
# ============================================================
y_log = np.log1p(y)


# ============================================================
# 7) KFold（交差検証）で学習して平均化
# ------------------------------------------------------------
# - 1回の学習だと分割の運でブレる
# - 5foldで test予測を平均すると安定することが多い
# ============================================================
kf = KFold(n_splits=5, shuffle=True, random_state=42)

oof_log = np.zeros(len(X))        # OOF（CV評価用）
test_pred = np.zeros(len(X_test)) # テスト予測の平均

for fold, (tr_idx, va_idx) in enumerate(kf.split(X), start=1):
    # --- split ---
    X_tr, X_va = X.iloc[tr_idx], X.iloc[va_idx]
    y_tr, y_va = y_log.iloc[tr_idx], y_log.iloc[va_idx]

    # CatBoost用のPool（カテゴリ列を指定）
    train_pool = Pool(X_tr, y_tr, cat_features=cat_idx)
    valid_pool = Pool(X_va, y_va, cat_features=cat_idx)
    test_pool  = Pool(X_test, cat_features=cat_idx)

    # =========================================================
    # 8) CatBoostモデル（depth=6用のおすすめ設定）
    # ---------------------------------------------------------
    # depth=6 で表現力が少し落ちるので
    # learning_rate を 0.03 に上げ、iterations を 30000 に下げる案（高速化も兼ねる）
    #
    # もし「学習が足りない」感じがあれば、iterationsを 50000 に戻してもOK。
    #
    # od_wait：
    #   改善が止まってから何回待つか。長すぎると無駄に時間が伸びる。
    #   depth=6では 400 くらいが現実的な妥協点になりやすい。
    # =========================================================
    model = CatBoostRegressor(
        loss_function="RMSE",
        eval_metric="RMSE",
        iterations=30000,
        learning_rate=0.03,
        depth=6,              # ★ここが今回の主役
        l2_leaf_reg=3.0,
        random_seed=42,
        od_type="Iter",
        od_wait=400,
        verbose=0
    )

    # 学習（use_best_model=True で検証スコアが最良のところを使う）
    model.fit(train_pool, eval_set=valid_pool, use_best_model=True)

    # --- validation prediction (log space) ---
    pred_va_log = model.predict(valid_pool)
    oof_log[va_idx] = pred_va_log

    # foldのRMSEを「ドル空間」で確認（提出スコアと同じ単位）
    pred_va_price = np.expm1(pred_va_log)
    y_va_price = np.expm1(y_va)
    fold_rmse = mean_squared_error(y_va_price, pred_va_price, squared=False)
    print(f"[Fold {fold}] RMSE(price): {fold_rmse:.2f}")

    # --- test prediction (average) ---
    pred_test_log = model.predict(test_pool)
    test_pred += np.expm1(pred_test_log) / kf.n_splits


# ============================================================
# 9) 全体CV RMSE（参考）
# ------------------------------------------------------------
# これは「目安」です。
# Public LB と完全一致はしませんが、方向性確認には役立ちます。
# ============================================================
cv_rmse = mean_squared_error(np.expm1(y_log), np.expm1(oof_log), squared=False)
print(f"\n[CV] RMSE(price): {cv_rmse:.2f}")


# ============================================================
# 10) 提出ファイル作成
# ============================================================
submission = pd.DataFrame({
    "Id": test["Id"],
    "SalePrice": test_pred
})
submission.to_csv("submission.csv", index=False)
print("✅ saved: submission.csv")


# ============================================================
# 実行時間が長いときの短縮メモ（必要ならここを変える）
# ------------------------------------------------------------
# 1) iterations を減らす（例：20000）
# 2) od_wait を減らす（例：300）
# 3) fold数を減らす（例：n_splits=3）
#
# ただし、fold数を減らすと予測がブレやすいので
# 最終提出では5fold推奨です。
# ============================================================
